<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\CategoryController;
use App\Http\Controllers\Api\SupplierController;
use App\Http\Controllers\Api\ProductController;
use App\Http\Controllers\Api\PurchaseController;
use App\Http\Controllers\Api\SaleController;
use App\Http\Controllers\Api\InvoiceController;
use App\Http\Controllers\Api\InventoryController;
use App\Http\Controllers\Api\RoleController;
use App\Http\Controllers\Api\UserController;
use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\DashboardController;
use App\Http\Controllers\Api\CustomerController;
use App\Http\Controllers\Api\SettingsController;
use App\Http\Controllers\Api\CashRegisterController;
use App\Http\Controllers\Api\ReportsController;

Route::get('/health', function () {
    return response()->json(['status' => 'ok', 'message' => 'API is running']);
});

Route::post('/auth/register', [AuthController::class, 'register']);
Route::post('/auth/login', [AuthController::class, 'login']);
Route::post('/auth/verify-password', [AuthController::class, 'verifyPassword']);

Route::middleware('auth:sanctum')->group(function () {
    Route::post('/auth/logout', [AuthController::class, 'logout']);
    Route::get('/auth/me', [AuthController::class, 'me']);
    Route::post('/auth/change-password', [AuthController::class, 'changePassword']);
    Route::post('/auth/profile-photo', [AuthController::class, 'updateProfilePhoto']);
    Route::post('/auth/tokens', [AuthController::class, 'createToken']);
    Route::get('/auth/tokens', [AuthController::class, 'listTokens']);
    Route::delete('/auth/tokens/{tokenId}', [AuthController::class, 'revokeToken']);

    Route::prefix('cash-register')->middleware('role:admin|seller')->group(function () {
        Route::get('/status', [CashRegisterController::class, 'status']);
        Route::post('/open', [CashRegisterController::class, 'open']);
        Route::post('/close', [CashRegisterController::class, 'close']);
        Route::get('/history', [CashRegisterController::class, 'history']);
        Route::get('/all-history', [CashRegisterController::class, 'allHistory'])->middleware('role:admin');
    });

    Route::prefix('roles')->middleware('role:admin')->group(function () {
        Route::get('/', [RoleController::class, 'index']);
        Route::post('/', [RoleController::class, 'store']);
        Route::get('/{role}', [RoleController::class, 'show']);
        Route::put('/{role}', [RoleController::class, 'update']);
        Route::delete('/{role}', [RoleController::class, 'destroy']);
    });

    Route::prefix('users')->middleware('role:admin')->group(function () {
        Route::get('/', [UserController::class, 'index']);
        Route::post('/', [UserController::class, 'store']);
        Route::get('/{user}', [UserController::class, 'show']);
        Route::put('/{user}', [UserController::class, 'update']);
        Route::delete('/{user}', [UserController::class, 'destroy']);
        Route::put('/{user}/toggle-status', [UserController::class, 'toggleStatus']);
        Route::post('/{user}/revoke-tokens', [UserController::class, 'revokeTokens']);
        Route::post('/{user}/change-password', [UserController::class, 'changePassword']);
    });

    Route::prefix('categories')->middleware('role:admin|warehouse')->group(function () {
        Route::get('/', [CategoryController::class, 'index']);
        Route::post('/', [CategoryController::class, 'store']);
        Route::get('/{category}', [CategoryController::class, 'show']);
        Route::put('/{category}', [CategoryController::class, 'update']);
        Route::delete('/{category}', [CategoryController::class, 'destroy']);
    });

    Route::prefix('suppliers')->middleware('role:admin|warehouse')->group(function () {
        Route::get('/', [SupplierController::class, 'index']);
        Route::post('/', [SupplierController::class, 'store']);
        Route::get('/{supplier}', [SupplierController::class, 'show']);
        Route::put('/{supplier}', [SupplierController::class, 'update']);
        Route::delete('/{supplier}', [SupplierController::class, 'destroy']);
    });

    Route::prefix('products')->group(function () {
        Route::get('/', [ProductController::class, 'index']);
        Route::get('/{product}', [ProductController::class, 'show']);

        Route::middleware('role:admin|warehouse')->group(function () {
            Route::post('/', [ProductController::class, 'store']);
            Route::put('/{product}', [ProductController::class, 'update']);
            Route::post('/{product}/image', [ProductController::class, 'uploadImage']);
            Route::delete('/{product}/image', [ProductController::class, 'deleteImage']);
            Route::delete('/{product}', [ProductController::class, 'destroy']);
        });
    });

    Route::prefix('purchases')->middleware('role:admin|warehouse')->group(function () {
        Route::get('/', [PurchaseController::class, 'index']);
        Route::post('/', [PurchaseController::class, 'store']);
        Route::get('/{purchase}', [PurchaseController::class, 'show']);
        Route::put('/{purchase}', [PurchaseController::class, 'update']);
        Route::delete('/{purchase}', [PurchaseController::class, 'destroy']);
        Route::put('/{purchase}/complete', [PurchaseController::class, 'complete']);
        Route::put('/{purchase}/cancel', [PurchaseController::class, 'cancel']);
    });

    Route::prefix('sales')->middleware('role:admin|seller')->group(function () {
        Route::get('/', [SaleController::class, 'index']);
        Route::post('/', [SaleController::class, 'store']);
        Route::get('/{sale}', [SaleController::class, 'show']);
        Route::put('/{sale}', [SaleController::class, 'update']);
        Route::delete('/{sale}', [SaleController::class, 'destroy']);
        Route::put('/{sale}/complete', [SaleController::class, 'complete']);
        Route::put('/{sale}/cancel', [SaleController::class, 'cancel']);
    });

    Route::prefix('invoices')->middleware('role:admin|seller')->group(function () {
        Route::get('/', [InvoiceController::class, 'index']);
        Route::post('/', [InvoiceController::class, 'store']);
        Route::get('/{invoice}', [InvoiceController::class, 'show']);
        Route::put('/{invoice}', [InvoiceController::class, 'update']);
        Route::delete('/{invoice}', [InvoiceController::class, 'destroy']);
        Route::put('/{invoice}/issue', [InvoiceController::class, 'issue']);
        Route::put('/{invoice}/paid', [InvoiceController::class, 'paid']);
        Route::put('/{invoice}/cancel', [InvoiceController::class, 'cancelInvoice']);
    });

    Route::prefix('customers')->group(function () {
        Route::get('/', [CustomerController::class, 'index']);
        Route::get('/by-document', [CustomerController::class, 'getByDocument']);
        Route::get('/{customer}', [CustomerController::class, 'show']);

        Route::middleware('role:admin|seller')->group(function () {
            Route::post('/', [CustomerController::class, 'store']);
            Route::put('/{customer}', [CustomerController::class, 'update']);
        });

        Route::middleware('role:admin')->group(function () {
            Route::delete('/{customer}', [CustomerController::class, 'destroy']);
        });
    });

    Route::prefix('inventories')->middleware('role:admin|warehouse')->group(function () {
        Route::get('/', [InventoryController::class, 'index']);
        Route::post('/adjust', [InventoryController::class, 'store']);
        Route::get('/{inventory}', [InventoryController::class, 'show']);
    });

    Route::prefix('dashboard')->group(function () {
        Route::get('/stats', [DashboardController::class, 'index']);
        Route::get('/recent-sales', [DashboardController::class, 'recentSales']);
        Route::get('/recent-purchases', [DashboardController::class, 'recentPurchases']);
        Route::get('/top-products', [DashboardController::class, 'topProducts']);
        Route::get('/low-stock', [DashboardController::class, 'lowStockProducts']);
        Route::get('/sales-chart', [DashboardController::class, 'salesChart']);
        Route::get('/purchases-chart', [DashboardController::class, 'purchasesChart']);
    });

    Route::prefix('settings')->group(function () {
        Route::get('/', [SettingsController::class, 'index']);
        Route::put('/', [SettingsController::class, 'update'])->middleware('role:admin');
        Route::post('/backup', [SettingsController::class, 'createBackup'])->middleware('role:admin|warehouse');
        Route::get('/backups', [SettingsController::class, 'listBackups'])->middleware('role:admin|warehouse');
        Route::get('/backup/download', [SettingsController::class, 'downloadBackup'])->middleware('role:admin|warehouse');
        Route::delete('/backup', [SettingsController::class, 'deleteBackup'])->middleware('role:admin|warehouse');
    });

    Route::prefix('reports')->middleware('role:admin|seller|warehouse')->group(function () {
        Route::get('/sales', [ReportsController::class, 'sales']);
        Route::get('/purchases', [ReportsController::class, 'purchases']);
        Route::get('/inventory', [ReportsController::class, 'inventory']);
        Route::get('/financial', [ReportsController::class, 'financial']);
    });
});
